
#include <maya/MPxFileTranslator.h>
#include <maya/MStringArray.h>
#include <maya/MGlobal.h>
#include <maya/MFnSkinCluster.h>

#include <MDt.h>
#include <MDtExt.h>

#include "rwcommon.h"
#include "material.h"
#include "remapper.h"
#include "rtimport.h"
#include "rpcollis.h"
#include "rtworld.h"

#include "querymel.h"
#include "mayabsp.h"
#include "blind.h"
#include "global.h"

static int numAddedVerts;
static int numAddedPolys;

int
bspTranslator::GetGroupTriCount(int shapeIndex, int groupIndex)
{
    int numFaceIndices, i, numFaces = 0;
    long *faceIndices;

    DtFaceGetIndexByShape(shapeIndex, groupIndex, &numFaceIndices, &faceIndices);

    /* Calculate numFaces from the face indices */
    i = 0;
    while (i<numFaceIndices)
    {
        i += 2;
        while (faceIndices[i] != -1)
        {
            numFaces++;
            i++;
        }
        i++;
    }

    return numFaces;
}

void
bspTranslator::GetPolygonsAndVerticesInShape(int shapeIndex, int *numPolys, int *numVerts)
{
    ReMapper reMapper;
    RwV3d **uvArrays = NULL;
    int numPolysInShape;
    int group;

    /* Find out the number of groups (materials) assigned to this shape */
    int numGroups = DtGroupGetCount(shapeIndex);

    numPolysInShape = 0;
    /* Loop through the groups and count the total faces */
    for (group = 0; group < numGroups; group++)
    {
        int numPolysInGroup;
        numPolysInGroup = GetGroupTriCount(shapeIndex, group);
        numPolysInShape += numPolysInGroup;
    }

    reMapper.SetNumInputFaces(numPolysInShape);

    /* Calculate arrays of transformed uv's ready to remap */
    uvArrays = (RwV3d **) RwMalloc(sizeof(RwV3d *) * numGroups);
    for (group = 0; group < numGroups; group++)
    {
        RwMatrix *trans;
        DtVec2f *tVerts;
        int count, i;
        DtGroupGetTextureVertices(shapeIndex, group, &count, &tVerts);
        if (count > 0)
        {
            uvArrays[group] = (RwV3d *) RwMalloc(sizeof(RwV3d) * count);
            trans = getUVTransformMatrix(shapeIndex, group);
            for (i=0; i<count; i++)
            {
                uvArrays[group][i].x = tVerts[i].vec[0];
                uvArrays[group][i].y = tVerts[i].vec[1];
                uvArrays[group][i].z = 0.0f;
            }
            free(tVerts);
            
            RwV3dTransformPoints(uvArrays[group], uvArrays[group], count, trans);

            /* Offset the texture coordinates to positive space and invert the Vs. */
            OffsetTextureCoords(uvArrays[group], count);
            
            RwMatrixDestroy(trans);
        }
        else
        {
            uvArrays[group] = NULL;
        }
    }

    /* Add all the groups into the remapper */
    for (group = 0; group < numGroups; group++)
    {
        reMapper.AddGroup(shapeIndex, group, globalData->materialMap,
            uvArrays[group], globalData->m_preLightWorld, globalData->m_limitUVs,
            globalData->m_uvLimit);
    }

    /* Destroy the uvArrays we created */
    for (group = 0; group < numGroups; group++)
    {
        if (uvArrays[group])
        {
            RwFree(uvArrays[group]);
        }
    }
    RwFree(uvArrays);

    /* Remap it all */
    reMapper.DoRemap();

    *numVerts = reMapper.GetNumVertices();
    *numPolys = reMapper.GetNumFaces();
}

void
bspTranslator::AddShapeToImportWorld(int shapeIndex, RtWorldImport *importWorld)
{
    ReMapper    reMapper;    
    int         numPolysInShape;
    int         group, count;
    int         numVerts, numFaces;
    int         i;
    MObject     object;
    char        *shapeName;
    DtVec3f     *mayaVerts;
    DtVec3f     *mayaNormals;
    RwV3d       **uvArrays = NULL;
    blindData   *shapeBlindData;
    ReMapperFaceLoop *remapFace;

    DtShapeGetVertices(shapeIndex, &count, &mayaVerts);
    if (DtShapeIsFlatShaded(shapeIndex))
    {
        DtShapeGetPolygonNormals(shapeIndex, &count, &mayaNormals);
    }
    else
    {
        DtShapeGetNormals(shapeIndex, &count, &mayaNormals);
    }

    // Find the name of the current shape that we are looking at.
    DtShapeGetName(shapeIndex, &shapeName);
    printf("Processing object named : %s\n", shapeName);

    /* Find out the number of groups (materials) assigned to this shape */
    int numGroups = DtGroupGetCount(shapeIndex);

    numPolysInShape = 0;
    /* Loop through the groups and count the total faces */
    for (group = 0; group < numGroups; group++)
    {
        int numPolysInGroup;        
        numPolysInGroup = GetGroupTriCount(shapeIndex, group);
        numPolysInShape += numPolysInGroup;
    }

    reMapper.SetNumInputFaces(numPolysInShape);

    /* Calculate arrays of transformed uv's ready to remap */
    uvArrays = (RwV3d **) RwMalloc(sizeof(RwV3d *) * numGroups);
    for (group = 0; group < numGroups; group++)
    {
        RwMatrix    *trans;
        DtVec2f     *tVerts;
        int         count, i;

        DtGroupGetTextureVertices(shapeIndex, group, &count, &tVerts);

        if (count > 0)
        {
            uvArrays[group] = (RwV3d *) RwMalloc(sizeof(RwV3d) * count);
            trans = getUVTransformMatrix(shapeIndex, group);
            for (i = 0; i < count; i++)
            {
                uvArrays[group][i].x = tVerts[i].vec[0];
                uvArrays[group][i].y = tVerts[i].vec[1];
                uvArrays[group][i].z = 0.0f;
            }
            free(tVerts);
            
            RwV3dTransformPoints(uvArrays[group], uvArrays[group], count, trans);

            /* Offset the texture coordinates to positive space and invert the Vs. */
            OffsetTextureCoords(uvArrays[group], count);
            
            RwMatrixDestroy(trans);
        }
        else
        {
            uvArrays[group] = NULL;
        }
    }

    /* Add all the groups into the remapper */
    for (group = 0; group < numGroups; group++)
    {
        reMapper.AddGroup(shapeIndex, group, globalData->materialMap, uvArrays[group],
            globalData->m_preLightWorld, globalData->m_limitUVs, globalData->m_uvLimit);
    }
    /* Remap it all */
    reMapper.DoRemap();

    numVerts = reMapper.GetNumVertices();
    numFaces = reMapper.GetNumFaces();

    /* Now add the data to our import world */
    RtWorldImportVertex     *verts = importWorld->vertices + numAddedVerts;
    RtWorldImportTriangle   *faces = importWorld->polygons + numAddedPolys;
     
    for (i = 0; i < numVerts; i++)
    {
        ReMapperVertex *remapVert = reMapper.GetVertex(i);

        verts[i].OC.x = mayaVerts[remapVert->vertexIndex].vec[0] * globalData->m_scaleFactor;
        verts[i].OC.y = mayaVerts[remapVert->vertexIndex].vec[1] * globalData->m_scaleFactor;
        verts[i].OC.z = mayaVerts[remapVert->vertexIndex].vec[2] * globalData->m_scaleFactor;

        if (mayaNormals)
        {
            verts[i].normal.x = mayaNormals[remapVert->normalIndex].vec[0];
            verts[i].normal.y = mayaNormals[remapVert->normalIndex].vec[1];
            verts[i].normal.z = mayaNormals[remapVert->normalIndex].vec[2];
        }

        verts[i].preLitCol.red      = remapVert->r;
        verts[i].preLitCol.green    = remapVert->g;
        verts[i].preLitCol.blue     = remapVert->b;
        verts[i].preLitCol.alpha    = remapVert->a;

        verts[i].texCoords.u = remapVert->u;
        verts[i].texCoords.v = remapVert->v;

        verts[i].pUserdata = (void *)&(globalData->blindDataRemap[globalData->worldBlindData->numVerts + i]);
    }

    /* Now the triangles */
    for (i = 0; i < numFaces; i++)
    {
        remapFace = reMapper.GetFace(i);

        faces[i].vertIndex[0]   = remapFace->index[0] + numAddedVerts;
        faces[i].vertIndex[1]   = remapFace->index[1] + numAddedVerts;
        faces[i].vertIndex[2]   = remapFace->index[2] + numAddedVerts;
        faces[i].matIndex       = globalData->materialMap[remapFace->materialIndex].rtImportMaterial;
        faces[i].pUserdata      = (void *)&(globalData->blindDataRemap[globalData->worldBlindData->numPolys + i]);
    }

    /* Get any blind data on this shape */
    DtExt_ShapeGetShapeNode(shapeIndex, object);
    shapeBlindData = new blindData(object);
    
    /* Remap it */
    shapeBlindData->remap(reMapper);

    /* Add the blind data to our world blind data */
    *globalData->worldBlindData += *shapeBlindData;

    /* Cleanup the shape blind data */
    delete (shapeBlindData);

    /* Destroy the uvArrays we created */
    for (group = 0; group < numGroups; group++)
    {
        if (uvArrays[group])
        {
            RwFree(uvArrays[group]);
        }
    }
    RwFree(uvArrays);

    numAddedVerts += numVerts;
    numAddedPolys += numFaces;
}

void
bspTranslator::AddSceneToImportWorld(RtWorldImport *importWorld)
{
    int numShapes = DtShapeGetCount();
    int vertsInWorld, polysInWorld;
    int shape;
    int i, maxIndex;

    vertsInWorld = 0;
    polysInWorld = 0;
    for (shape = 0; shape < numShapes; shape++)
    {
        int numPolys = 0;
        int numVerts = 0;
        
        GetPolygonsAndVerticesInShape(shape, &numPolys, &numVerts);
        
        polysInWorld += numPolys;
        vertsInWorld += numVerts;
    }

    RtWorldImportAddNumTriangles(importWorld, polysInWorld);
    RtWorldImportAddNumVertices(importWorld, vertsInWorld);
    
    maxIndex = (polysInWorld > vertsInWorld) ? polysInWorld : vertsInWorld;

    globalData->blindDataRemap = (int *)RwMalloc(maxIndex * sizeof(int));
    
    if (globalData->blindDataRemap == NULL)
    {
        return;
    }

    for (i = 0; i < maxIndex; i++)
    {
        globalData->blindDataRemap[i] = i;
    }

    for (shape = 0; shape < numShapes; shape++)
    {
        AddShapeToImportWorld(shape, importWorld);
    }
}

RwBool
worldProgressCallBack(RwInt32 msg, RwReal value)
{
    static int lastval = -1;
    switch(msg)
    {
        case rtWORLDIMPORTPROGRESSBSPBUILDSTART:
            printf("Starting World Sector Generation\n");
            lastval = -1;
            break;
        case rtWORLDIMPORTPROGRESSBSPBUILDUPDATE:
            if ((int)value > lastval)
            {
                printf("Generating World Sectors: %d%% done\n", (int)value);
                lastval = (int)value;
            }
            break;
        case rtWORLDIMPORTPROGRESSBSPBUILDEND:
            printf("World Sector Generation Completed\n");
            break;
        case rtWORLDIMPORTPROGRESSBSPCOMPRESSSTART:
            printf("Starting World Sector Compression\n");
            lastval = -1;
            break;
        case rtWORLDIMPORTPROGRESSBSPCOMPRESSUPDATE:
            if ((int)value > lastval)
            {
                printf("Compressing World Sectors: %d%% done\n", (int)value);
                lastval = (int)value;
            }
            break;
        case rtWORLDIMPORTPROGRESSBSPCOMPRESSEND:
            printf("World Sector Compression Completed\n");
            break;
    }
    return TRUE;
}


/*
    main function for the plugin
 */
MStatus 
bspTranslator::exportbsp( char *fileName )
{
    MStatus                     status = MStatus::kSuccess;
    RpWorld                     *world;
    bool                        exportBlindData = false;
    blindDataTemplateIt         bdTemplateIt;
    blindDataSectorRemapListIt  bdSectorIt;
    RtWorldImportParameters     importParams;
    RtWorldImport               *importWorld;

    /* Check all the file nodes in the scene and see if any have K&L data attached */
    forAllNodesInScene(MFn::kFileTexture, fileTextureCallBack, (void *)&globalData->m_exportKL);

    /* Check if the scene has any blind data templates */
    processBlineDataTemplates();
    
    if (globalData->blindDataTemplates.size() > 0)
    {
        exportBlindData = true;
    }

    OpenRenderWare(NULL, false, false, false, true, false, false, globalData->m_exportKL, exportBlindData);

    /* Initialize the Dt database */

    printf("Initialising Maya SDK\n");

    DtExt_SceneInit("exportbsp");
    DtExt_setOutputCameras(false);
    DtExt_setMultiTexture(false);

    /* Disabling inline textures causes a crash on Maya 2.0 */
#ifndef MAYA_20
    DtExt_setInlineTextures(false);
#endif

    /* Let's have DT collapse all the transforms, and
       transform the vertices */
    DtExt_setOutputTransforms(kTRANSFORMNONE);

    /* Walk the dag and fill in the internal database */
    DtExt_dbInit();    

    printf("Finished initialising Maya SDK\n");

    /* Setup an import world with our parameters */
    importWorld = RtWorldImportCreate();
    RtWorldImportParametersInitialize(&importParams);
    importParams.worldSectorMaxSize         = globalData->m_maxSectorSize;
    importParams.maxWorldSectorPolygons     = globalData->m_maxPolysPerSector;
    importParams.maxClosestCheck            = globalData->m_maxClosestCheck;
    importParams.weldThreshold              = globalData->m_weldThreshold;
    importParams.maxOverlapPercent          = globalData->m_maxSectorOverlap;
    importParams.noAlphaInOverlap           = globalData->m_noAlphaInOverlap;
    importParams.calcNormals                = false;
    importParams.conditionGeometry          = false;
    importParams.generateCollTrees          = true;
    if (globalData->m_triStrips)
    {
        importParams.flags |= rpWORLDTRISTRIP;
    }
    importParams.flags |= rpWORLDMODULATEMATERIALCOLOR;

    if (!globalData->m_exportNormals)
    {
        importParams.flags &= ~rpWORLDNORMALS;
    }

    /* Create a scene wide material mapper */
    globalData->materialMap = BuildMaterialMap(importWorld, globalData->m_verbose,
        &globalData->m_textureNameWarning);

    /* Create a scene blind data place holder */
    globalData->worldBlindData = new blindData();

    /* Setup the import blind data callbacks */
    if (globalData->blindDataTemplates.size() > 0)
    {
        RtWorldImportSetUserdataCallBacks(NULL, NULL, interpVertexCB,
                            setVertexCB, NULL, NULL, setPolygonCB);
    }

    /* Build the import world */
    numAddedVerts = 0;
    numAddedPolys = 0;

    AddSceneToImportWorld(importWorld);

    RtWorldImportSetProgressCallBack(worldProgressCallBack);

    /* Now convert to an rpworld */
    world = RtWorldImportCreateWorld(importWorld, &importParams);
    RtWorldImportDestroy(importWorld);

    RtWorldImportSetProgressCallBack(NULL);

    if (world)
    {
        printf("World has %d polygons, %d vertices and %d sectors\n",
            RtWorldGetNumPolygons(world), RtWorldGetNumVertices(world),
            RtWorldGetNumWorldSectors(world));

        if (globalData->m_generateCollision)
        {
            RpCollisionWorldBuildData(world, NULL);
        }

        if (WorldHasColoredMaterials(world))
        {
            RpWorldSetFlags(world, RpWorldGetFlags(world) | rpWORLDMODULATEMATERIALCOLOR);
        }

        RpWorldForAllWorldSectors(world, WorldSectorHasMaterialFX, NULL);

        /* Add our blind data to the world sectors */
        RpWorldForAllWorldSectors(world, globalData->worldBlindData->addToWorldSector,
            globalData->worldBlindData);
    }

    /* If converted successfully write it out */
    if (world)
    {
        RwStream *stream = RwStreamOpen(rwSTREAMFILENAME, rwSTREAMWRITE, fileName);

        RpWorldStreamWrite(world, stream);
        RwStreamClose(stream, NULL);
        RpWorldDestroy(world);
    }
    else
    {
        printf("Failed to convert import world to RpWorld\n");
        status = MStatus::kFailure;
    }

    /* Clean up the allocated memory and internal storage */
    if (globalData->materialMap != NULL)
    {
        DestroyMaterialMap(globalData->materialMap);
        globalData->materialMap = NULL;
    }
    
    /* free the blind data template names */
    for (bdTemplateIt = globalData->blindDataTemplates.begin();
            bdTemplateIt != globalData->blindDataTemplates.end();
            bdTemplateIt++)
    {
        free (bdTemplateIt->templateNodeName);
        free (bdTemplateIt->templateName);
    }

    /* And the world sector remaps */ 
    for (bdSectorIt = globalData->sectorMap.begin();
            bdSectorIt != globalData->sectorMap.end();
            bdSectorIt++)
    {
        blindDataSectorRemap *remap = *bdSectorIt;
        
        if (remap->vertexMap != NULL)
        {
            free (remap->vertexMap);
        }
        if (remap->polygonMap != NULL)
        {
            free (remap->polygonMap);
        }
    }
    

    if (globalData->worldBlindData != NULL)
    {
        delete (globalData->worldBlindData);
    }

    if (globalData->blindDataRemap != NULL)
    {
        RwFree(globalData->blindDataRemap);
    }

    DtExt_CleanUp();
    
    CloseRenderWare();

    printf("Export completed %s\n", status == MStatus::kSuccess ? "" : "with errors");

    /* return from the plugin command */
    return status;
}
